import torch.nn as nn
import torch
from timm.models.layers import DropPath
from spikingjelly.clock_driven.neuron import (
    MultiStepLIFNode,
    MultiStepParametricLIFNode,
)


class Erode(nn.Module):
    def __init__(self) -> None:
        super().__init__()
        self.pool = nn.MaxPool3d(
            kernel_size=(1, 3, 3), stride=(1, 1, 1), padding=(0, 1, 1)
        )

    def forward(self, x):
        return self.pool(x)


class MS_MLP_Conv(nn.Module):
    def __init__(
        self,
        in_features,
        hidden_features=None,
        out_features=None,
        drop=0.0,
        spike_mode="lif",
        layer=0,
    ):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = in_features // 3 * 16
        self.res = in_features == hidden_features
        self.fc1_conv = nn.Conv2d(in_features, hidden_features, kernel_size=1, stride=1)
        self.fc1_bn = nn.BatchNorm2d(hidden_features)
        if spike_mode == "lif":
            self.fc1_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        elif spike_mode == "plif":
            self.fc1_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )
        
        self.dw_conv = nn.Conv2d(hidden_features // 2, hidden_features // 2, kernel_size=3, stride=1, padding=1, groups=hidden_features // 2, bias=True)
        self.dw_bn = nn.BatchNorm2d(hidden_features // 2)
        self.dw_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.fc2_conv = nn.Conv2d(
            hidden_features // 2, out_features, kernel_size=1, stride=1
        )
        self.fc2_bn = nn.BatchNorm2d(out_features)
        if spike_mode == "lif":
            self.fc2_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        elif spike_mode == "plif":
            self.fc2_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )

        self.c_hidden = hidden_features
        self.c_output = out_features
        self.layer = layer

    def forward(self, x, hook=None):
        T, B, C, H, W = x.shape
        identity = x

        x = self.fc1_lif(x)
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_fc1_lif"] = x.detach()
        x = self.fc1_conv(x.flatten(0, 1))
        x = self.fc1_bn(x).reshape(T, B, self.c_hidden, H, W).contiguous()
        if self.res:
            x = identity + x
            identity = x
        x = self.fc2_lif(x)
        x1, x2 = torch.chunk(x, 2, dim=2)
        x1 = self.dw_conv(x1.flatten(0, 1))
        x1 = self.dw_bn(x1).reshape(T,B,self.c_hidden // 2, H, W)
        x1 = self.dw_lif(x1)
        x = x1 * x2
        
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_fc2_lif"] = x.detach()
        x = self.fc2_conv(x.flatten(0, 1))
        x = self.fc2_bn(x).reshape(T, B, C, H, W).contiguous()

        x = x + identity
        return x, hook
    

class Expert_Unit(nn.Module):
    def __init__(self, in_features, out_features, bias=True):
        super().__init__()
        self.unit_conv = nn.Conv2d(in_features, out_features, kernel_size=1, stride=1, bias=bias)
        self.unit_bn = nn.BatchNorm2d(out_features)
        self.unit_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.out_features = out_features

    def forward(self, x, hook=None):
        T, B, C, H, W = x.shape
        x = self.unit_conv(x.flatten(0, 1))
        x = self.unit_bn(x).reshape(T, B, self.out_features, H, W).contiguous()
        x = self.unit_lif(x)
        return x, hook


class MS_SSA_Conv(nn.Module):
    def __init__(
        self,
        dim,
        num_heads=8,
        head_dim=0,
        qkv_bias=False,
        qk_scale=None,
        attn_drop=0.0,
        proj_drop=0.0,
        sr_ratio=1,
        mode="direct_xor",
        spike_mode="lif",
        dvs=False,
        layer=0,
        num_expert=4,
    ):
        super().__init__()
        assert (
            dim % num_heads == 0
        ), f"dim {dim} should be divided by num_heads {num_heads}."
        self.dim = dim
        self.dvs = dvs
        self.num_heads = num_heads
        self.head_dim = head_dim
        self.num_expert = num_expert
        if dvs:
            self.pool = Erode()
        self.scale = 0.125
        '''
        self.q_conv = nn.Conv2d(dim, head_dim, kernel_size=1, stride=1, bias=False)
        self.q_bn = nn.BatchNorm2d(head_dim)
        if spike_mode == "lif":
            self.q_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        elif spike_mode == "plif":
            self.q_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )
        '''
        
        self.k_conv = nn.Conv2d(dim, head_dim, kernel_size=1, stride=1, bias=False)
        self.k_bn = nn.BatchNorm2d(head_dim)
        if spike_mode == "lif":
            self.k_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        elif spike_mode == "plif":
            self.k_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )

        self.v_conv = nn.Conv2d(dim, head_dim, kernel_size=1, stride=1, bias=False)
        self.v_bn = nn.BatchNorm2d(head_dim)
        if spike_mode == "lif":
            self.v_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend="cupy")
        elif spike_mode == "plif":
            self.v_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )

        self.talking_heads = nn.Conv1d(
            num_heads, num_heads, kernel_size=1, stride=1, bias=False
        )
        if spike_mode == "lif":
            self.talking_heads_lif = nn.ModuleList([MultiStepLIFNode(
                tau=2.0, v_threshold=0.5, detach_reset=True, backend="cupy"
            ) for i in range(num_expert)])
        elif spike_mode == "plif":
            self.talking_heads_lif = MultiStepParametricLIFNode(
                init_tau=2.0, v_threshold=0.5, detach_reset=True, backend="cupy"
            )
        
        self.router1 = nn.Conv2d(dim, num_expert, kernel_size=1, stride=1)
        self.router2 = nn.BatchNorm2d(num_expert)
        self.router3 = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')
        
        self.ff_list = nn.ModuleList([Expert_Unit(dim, head_dim, False) for i in range(num_expert)])


        self.proj_conv = nn.Conv2d(head_dim, dim, kernel_size=1, stride=1)
        self.proj_bn = nn.BatchNorm2d(dim)

        if spike_mode == "lif":
            self.shortcut_lif = MultiStepLIFNode(
                tau=2.0, detach_reset=True, backend="cupy"
            )
        elif spike_mode == "plif":
            self.shortcut_lif = MultiStepParametricLIFNode(
                init_tau=2.0, detach_reset=True, backend="cupy"
            )

        self.mode = mode
        self.layer = layer

    def forward(self, x, hook=None):
        T, B, C, H, W = x.shape
        identity = x
        N = H * W
        x = self.shortcut_lif(x)
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_first_lif"] = x.detach()

        x_for_qkv = x.flatten(0, 1)
        '''
        q_conv_out = self.q_conv(x_for_qkv)
        q_conv_out = self.q_bn(q_conv_out).reshape(T, B, C, H, W).contiguous()
        q_conv_out = self.q_lif(q_conv_out)
        
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_q_lif"] = q_conv_out.detach()
        q = (
            q_conv_out.flatten(3)
            .transpose(-1, -2)
            .reshape(T, B, N, self.num_heads, C // self.num_heads)
            .permute(0, 1, 3, 2, 4)
            .contiguous()
        )
        '''

        k_conv_out = self.k_conv(x_for_qkv)
        k_conv_out = self.k_bn(k_conv_out).reshape(T, B, self.head_dim, H, W).contiguous()
        k_conv_out = self.k_lif(k_conv_out)
        if self.dvs:
            k_conv_out = self.pool(k_conv_out)
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_k_lif"] = k_conv_out.detach()
        k = k_conv_out.flatten(3).transpose(-1, -2).contiguous()  # T B N head_dim
        '''
        k = (
            k_conv_out.flatten(3)
            .transpose(-1, -2)
            .reshape(T, B, N, self.num_heads, C // self.num_heads)
            .permute(0, 1, 3, 2, 4)
            .contiguous()
        )
        '''

        v_conv_out = self.v_conv(x_for_qkv)
        v_conv_out = self.v_bn(v_conv_out).reshape(T, B, self.head_dim, H, W).contiguous()
        v_conv_out = self.v_lif(v_conv_out)
        '''
        if self.dvs:
            v_conv_out = self.pool(v_conv_out)
        '''
        if hook is not None:
            hook[self._get_name() + str(self.layer) + "_v_lif"] = v_conv_out.detach()
        v = v_conv_out.flatten(3).transpose(-1, -2).contiguous()  # T B N C
        '''
        v = (
            v_conv_out.flatten(3)
            .transpose(-1, -2)
            .reshape(T, B, N, self.num_heads, C // self.num_heads)
            .permute(0, 1, 3, 2, 4)
            .contiguous()
        )  # T B head N C//h
        '''
        
        weights = self.router1(x_for_qkv)
        weights = self.router2(weights).reshape(T, B, self.num_expert, H, W).contiguous()
        weights = self.router3(weights)
        
        y = 0
        for idx in range(self.num_expert):
            weight_idx = weights[:, :, idx, :, :].unsqueeze(dim=-3)
            q_conv_out, hook = self.ff_list[idx](x)
            if self.dvs:
                q_conv_out = self.pool(q_conv_out)                                 
            q = q_conv_out.flatten(3).transpose(-1, -2).contiguous()
            qk = k.mul(q) # T B N head_dim
            if hook is not None:
                hook[self._get_name() + str(self.layer) + 'expert_' + str(idx) + "_kv_before"] = qk
            if self.dvs:
                qk = self.pool(qk)
            qk = qk.sum(dim=-2, keepdim=True)
            qk = self.talking_heads_lif[idx](qk)
            if hook is not None:
                hook[self._get_name() + str(self.layer) + 'expert_' + str(idx) + "_kv"] = qk.detach()
            result = v.mul(qk)
            if self.dvs:
                result = self.pool(result)
            if hook is not None:
                hook[self._get_name() + str(self.layer) + "_x_after_qkv"] = x.detach()
            y += weight_idx * result.transpose(-1, -2).reshape(T, B, self.head_dim, H, W).contiguous()
                                                   
        y = (
            self.proj_bn(self.proj_conv(y.flatten(0, 1)))
            .reshape(T, B, C, H, W)
            .contiguous()
        )

        y = y + identity
        return y, v, hook


class MS_Block_Conv(nn.Module):
    def __init__(
        self,
        dim,
        num_heads,
        head_dim,
        mlp_ratio=4.0,
        qkv_bias=False,
        qk_scale=None,
        drop=0.0,
        attn_drop=0.0,
        drop_path=0.0,
        norm_layer=nn.LayerNorm,
        sr_ratio=1,
        attn_mode="direct_xor",
        spike_mode="lif",
        dvs=False,
        layer=0,
        num_expert=4,
    ):
        super().__init__()
        self.attn = MS_SSA_Conv(
            dim,
            num_heads=num_heads,
            head_dim=head_dim,
            qkv_bias=qkv_bias,
            qk_scale=qk_scale,
            attn_drop=attn_drop,
            proj_drop=drop,
            sr_ratio=sr_ratio,
            mode=attn_mode,
            spike_mode=spike_mode,
            dvs=dvs,
            layer=layer,
            num_expert=num_expert,
        )
        self.drop_path = DropPath(drop_path) if drop_path > 0.0 else nn.Identity()
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = MS_MLP_Conv(
            in_features=dim,
            hidden_features=mlp_hidden_dim,
            drop=drop,
            spike_mode=spike_mode,
            layer=layer,
        )

    def forward(self, x, hook=None):
        x_attn, attn, hook = self.attn(x, hook=hook)
        x, hook = self.mlp(x_attn, hook=hook)
        return x, attn, hook
